<?php

class WC_OPAY_Gateway extends WC_Payment_Gateway
{
    public $wc;
    public $wp_language;
    public $opay;
    public $website_id;
    public $test_mode;
    public $user_id;
    public $signature_type;
    public $private_key;
    public $opay_certificate;
    public $signature_password;
    public $show_payment_methods;
    public $height_of_banks_icons;
    public $enabled_wc_blocks;
    public $enabled_opay_everywhere;
    public $use_remote_website_settings;
    const APP_NAME = 'Woocommerce';
    const WIDGET_VERSION = '1.5.15';
    const CONFIG_COUNTRIES_ALL = ['default' => 'LT', 'LV', 'EE'];
    const CONFIG_COUNTRIES_SETTINGS = [
        'LT' => ['languages' => ['default' => 'lit', 'rus', 'eng', 'lav', 'est']],
        'LV' => ['languages' => ['default' => 'lav', 'rus', 'eng', 'lit', 'est']],
        'EE' => ['languages' => ['default' => 'est', 'rus', 'eng', 'lit', 'lav']],
        'OTHER' => ['languages' => ['default' => 'eng', 'rus', 'lit', 'lav', 'est']],
    ];

    public function __construct()
    {
        // Older WooCommerce version doesn't set WC function
        if (function_exists('WC')) {
            $this->wc = WC();
        } elseif (isset($GLOBALS['woocommerce'])) {
            $this->wc = $GLOBALS['woocommerce'];
        }

        // Sets system language
        $this->wp_language = substr(get_locale(), 0, 2);

        // The global ID for this Payment method
        $this->id = 'opay';
        if (is_admin()) {
            // The Title shown on the top of the Payment Gateways Page next to all the other Payment Gateways
            $this->method_title = __('OPAY Payment Gateway', 'opay-woocommerce');
        } else {
            // The Title shown on the top of the Payment Gateways Page next to all the other Payment Gateways
            $this->method_title = ($this->get_settings('payment_gateway_name') != '') ? $this->get_settings('payment_gateway_name') : __('OPAY Payment Gateway', 'opay-woocommerce');
        }

        // The title to be used for the vertical tabs that can be ordered top to bottom
        $this->title = ($this->get_settings('payment_gateway_name') != '') ? $this->get_settings('payment_gateway_name') : __('OPAY Payment Gateway', 'opay-woocommerce');


        // The description for this Payment Gateway, shown on the actual Payment options page on the backend
        $this->method_description = __('OPAY Payment Gateway plugin for WooCommerce', 'opay-woocommerce');


        // URL of an image next to the gateway's name on the frontend
        $this->icon = null;

        // when doing a direct integration (when CC fields are integrated in)
        $this->has_fields = true;
        $this->supports = array('default_credit_card_form');


        // This basically defines your settings which are then loaded with init_settings()
        $this->init_form_fields();

        // After init_settings() is called, you can get the settings and load them into variables, e.g:
        // $this->title = $this->get_option( 'title' );
        $this->init_settings();

        // Turn these settings into variables we can use
        foreach ($this->settings as $setting_key => $value) {
            $this->$setting_key = $value;
        }

        if (is_admin()) {
            if (version_compare(WOOCOMMERCE_VERSION, '2.0.0', '>=')) {
                add_action('woocommerce_update_options_payment_gateways_' . $this->id, array(&$this, 'process_admin_options'));
            } else {
                add_action('woocommerce_update_options_payment_gateways', array(&$this, 'process_admin_options'));
            }

            add_action('admin_enqueue_scripts', array($this, 'add_admin_javascript'));
        } else {
            add_action('woocommerce_thankyou_' . $this->id, array($this, 'handle_request_to_redirect_url'));
        }

        //emisija.lt
        if (is_checkout()) {
            add_action('wp_head', array($this, 'add_checkout_css'));
        }

        //Order refund initiate request
        add_action('woocommerce_order_refunded', array($this, 'em_handle_request_to_refund_order'), 999, 2);
        add_action('em_opay_refunded_order_status_checker', array($this, 'em_opay_refunded_order_status_checker_function'));
        //

        add_action('woocommerce_api_wc_opay_gateway', array($this, 'handle_request_to_web_service_url'));
    }

    public function get_title()
    {
        /**
         * Title field could get translated by WCML on load. Since translations
         * for this field come from our side, we prevent that.
         */
        $paymentGatewayName = ($this->get_settings('payment_gateway_name') != '') ? $this->get_settings('payment_gateway_name') : __('OPAY Payment Gateway', 'opay-woocommerce');
        $this->title = $this->title != $paymentGatewayName ? $paymentGatewayName : $this->title;
        return $this->title;
    }

    //emisija.lt - Function to check order status
    public function em_opay_refunded_order_status_checker_function()
    {
        $website_id = $this->website_id;

        //Get all Opay refunded Order's
        $args = array(
            'meta_key' => 'opay_refund_status',
            'meta_value' => 'initiate',
            'meta_compare' => '=',
            'return' => 'ids'
        );

        $orders = wc_get_orders($args);

        if (!empty($orders)) {
            foreach ($orders as $order_id) {
                $order = new WC_Order($order_id);

                //Check if order has unique refund token
                $unique_refund_token = $order->get_meta('opay_unique_refund_token', true);
                $refund_type = $order->get_meta('refund_type', true);

                if (empty($unique_refund_token)) {
                    continue;
                }

                $lang = get_bloginfo("language");
                $lan = $this->getLanguage($lang);

                $params = [
                    'language' => $lan,
                    'service_name' => 'getRefundRequestStatus',
                    'unique_refund_token' => $unique_refund_token,
                    'website_id' => $website_id,
                ];

                $stringToBeSigned = '';
                foreach ($params as $key => $val) {
                    // http_build_query strips parameters which have null values, so we do the same here (normally you shouldn't pass parameters with null values here)
                    if (is_null($val)) {
                        continue;
                    }

                    if (is_bool($val)) {
                        $val = (int) $val;
                    }

                    $stringToBeSigned .= $key . $val;
                }

                $opay_signature_password = md5($stringToBeSigned . $this->signature_password);
                $params['password_signature'] = $opay_signature_password;
                $url = 'https://gateway.opay.lt/api/refunds/';

                $request = $this->opay->sendRequest($url, 'POST', $params);

                $data = json_decode($request, true);

                //Check if Order refund status is approved
                if ($data['response']['result']['status'] == 'approved' && $refund_type == 'full') {
                    $order->update_meta_data('opay_refund_status', 'approved');

                    // The text for the note
                    $note = __("Full refund completed. Funds successfully returned to the customer.", "opay-woocommerce");

                    // Add the note
                    $order->add_order_note($note);

                    $order->update_status('refunded');

                    $order->save();
                } elseif ($data['response']['result']['status'] == 'approved' && $refund_type == 'partial') {
                    $order->update_meta_data('opay_refund_status', 'partial-approved');

                    // The text for the note
                    $note = __("Partial refund completed. Funds successfully returned to the customer.", "opay-woocommerce");

                    // Add the note
                    $order->add_order_note($note);

                    $order->save();
                }
            }
        }
    }

    //emisija.lt - Function to initiate order refund
    public function em_handle_request_to_refund_order($order_id, $refund_id)
    {
        try {
            if (!isset($this->opay)) {
                require_once dirname(__FILE__) . '/lib/opay_8.1.gateway.inc.php';
                $this->opay = new OpayGateway();
            }

            $order = new WC_Order($order_id);

            $website_id = $this->website_id;

            //Get Order payment token
            $opay_payment_token = $order->get_meta('opay_payment_token', true);

            //Get Order refund token
            $unique_refund_token = $order->get_meta('opay_unique_refund_token', true);
            $refund_type = $order->get_meta('refund_type', true);
            $refund_amount = $order->get_meta('refund_amount', true);

            //Check if the order has already been refundend
            if (!empty($unique_refund_token) && $refund_type == 'full') {
                //Do nothing
            } else {
                //Get & Set default language
                $lang = get_bloginfo("language");
                //Get Order Currency
                $currency = $order->get_currency();
                $lan = $this->getLanguage($lang);

                //Set Refund type (full or partial)
                $order_refund_type = 'full';

                //Get Order Refund amount
                if ($order->get_total_refunded() == $order->get_total()) {
                    //Fully
                    $amount = $order->get_total_refunded();
                    $order_refund_type = 'full';
                } else {
                    //Partial
                    $amount = $order->get_total_refunded();
                    $order_refund_type = 'partial';
                }

                //Convert amount to decimals (cents)
                $amount = bcmul($amount, 100);

                if (!empty($refund_amount)) {
                    $amount = $amount - $refund_amount;
                }

                //Set Refund Params
                $params = [
                    'language' => $lan,
                    'service_name' => 'refundRequest',
                    'unique_payment_token' => $opay_payment_token,
                    'amount' => $amount,
                    'currency' => $currency,
                    'website_id' => $website_id,
                ];

                //Set Opay signature password & url
                $stringToBeSigned = '';
                foreach ($params as $key => $val) {
                    // http_build_query strips parameters which have null values, so we do the same here (normally you shouldn't pass parameters with null values here)
                    if (is_null($val)) {
                        continue;
                    }

                    if (is_bool($val)) {
                        $val = (int) $val;
                    }

                    $stringToBeSigned .= $key . $val;
                }

                $opay_signature_password = md5($stringToBeSigned . $this->signature_password);
                $params['password_signature'] = $opay_signature_password;
                $url = 'https://gateway.opay.lt/api/refunds/';

                $request = $this->opay->webServiceRequest($url, $params);

                if ($request['response']['result']) {
                    //Add Opay succesfull refund token
                    $order->update_meta_data('opay_unique_refund_token', $request['response']['result']['unique_refund_token']);
                    $order->update_meta_data('refund_type', $order_refund_type);
                    $order->update_meta_data('opay_refund_status', 'initiate');

                    $order->update_meta_data('refund_amount', bcmul($order->get_total_refunded(), 100));

                    // The text for the note
                    $note = __("Refund created. Log in to OPAY <a href='https://gateway.opay.lt/my/'>self-service</a> to approve the return of funds.", "opay-woocommerce");

                    // Add the note
                    $order->add_order_note($note);
                } else {
                    // The text for the note
                    $note = __("Order refund failed. Error occurred.", "opay-woocommerce");
                    $order->update_meta_data('opay_refund_status', 'error');
                    // Add the note
                    $order->add_order_note($note);
                }

                $order->save();
            }
        } catch (OpayGatewayException $e) {
            // do nothing
        }
    }

    protected function set_order_complete()
    {
        try {
            if (!isset($this->opay)) {
                $this->opay = new OpayGateway();
            }

            $parametersArray = array();

            if (isset($_POST['encoded'])) {
                $parametersArray = $this->opay->convertEncodedStringToArrayOfParameters($_POST['encoded']);
            } else if (isset($_GET['encoded'])) {
                $parametersArray = $this->opay->convertEncodedStringToArrayOfParameters($_GET['encoded']);
            } else if (isset($_POST['password_signature']) || isset($_POST['rsa_signature'])) {
                $parametersArray = $_POST;
            } else if (isset($_GET['password_signature']) || isset($_GET['rsa_signature'])) {
                $parametersArray = $_GET;
            }

            if (!empty($parametersArray)) {
                // Dublicate status triggers fix - Transient lock
                $orderId = $this->get_order_id_by_order_number($parametersArray['order_nr']);
                $lock_key = 'opay_processing_' . $orderId;

                if (get_transient($lock_key)) {
                    return; // Other process already running this function
                }

                // Set lock for 1 minute
                set_transient($lock_key, time(), 60);

                if ($this->signature_type == 'rsa_private_key') {
                    $this->opay->setMerchantRsaPrivateKey($this->private_key);
                    $this->opay->setOpayCertificate($this->opay_certificate);
                } else {
                    $this->opay->setSignaturePassword($this->signature_password);
                }

                if ($this->opay->verifySignature($parametersArray)) {
                    if ($parametersArray['status'] == '1') {
                        global $wpdb;
                        $request = "SELECT ENGINE FROM information_schema.TABLES WHERE TABLE_NAME = '{$wpdb->posts}' AND TABLE_SCHEMA = '{$wpdb->dbname}'";
                        $result = $wpdb->get_row($request);
                        $engine = isset($result->ENGINE) ? $result->ENGINE : '';

                        // RACE CONDITION FIX: Get proper order ID first
                        $order = new WC_Order($orderId);
                        $opay_payment_token = $order->get_meta('opay_payment_token', true);

                        if (empty($opay_payment_token)) {
                            //Add Opay succesfull payment token - emisija.lt
                            $order->update_meta_data('opay_payment_token', $parametersArray['p_token']);
                        }

                        // if such order exists
                        if (isset($order->billing_country)) {
                            if ($engine == 'InnoDB') {
                                // FIXED: Use proper post_id for locking instead of order_nr
                                $wpdb->query('START TRANSACTION');

                                // FIXED: Use proper post_id for locking instead of order_nr
                                $lockedOrder = $wpdb->get_row($wpdb->prepare("SELECT ID, post_status FROM {$wpdb->posts} WHERE ID = %d AND post_type = 'shop_order' FOR UPDATE", $orderId));

                                if (
                                    isset($lockedOrder->post_status) &&
                                    $lockedOrder->post_status != 'wc-pending' &&
                                    $lockedOrder->post_status != 'wc-cancelled' &&
                                    $lockedOrder->post_status != 'wc-failed'
                                ) {
                                    $wpdb->query('ROLLBACK');
                                    return;
                                }
                            }

                            if (method_exists($order, 'get_currency')) {
                                $currency = strtoupper($order->get_currency());
                            } elseif (method_exists($order, 'get_order_currency')) {
                                $currency = strtoupper($order->get_order_currency());
                            } else {
                                $currency = strtoupper(get_woocommerce_currency());
                            }

                            if ($currency == strtoupper($parametersArray['p_currency'])) {
                                $orderTotal = intval(number_format($order->get_total(), 2, '', ''));
                                if ($parametersArray['p_amount'] >= $orderTotal) {
                                    if (method_exists($order, 'get_status')) {
                                        $orderStatus = strtolower($order->get_status());
                                    } else {
                                        $orderStatus = strtolower($order->status);
                                    }
                                    if (in_array($orderStatus, array('pending', 'cancelled', 'failed'))) {
                                        $order->payment_complete();
                                    }
                                }
                            }

                            if ($engine == 'InnoDB') {
                                $wpdb->query('COMMIT');
                            }
                        }
                    }
                }
            }
        } catch (OpayGatewayException $e) {
            // do nothing
        }
    }

    public function handle_request_to_redirect_url($orderId)
    {
        global $woocommerce;
        require_once dirname(__FILE__) . '/lib/opay_8.1.gateway.inc.php';

        $this->set_order_complete();
        $woocommerce->cart->empty_cart();
    }

    public function handle_request_to_web_service_url()
    {
        require_once __DIR__ . '/lib/opay_8.1.gateway.inc.php';

        // request came from checkout page. Here we redirect customer to OPAY
        if (!empty($_REQUEST['redirect-to-opay']) && !empty($_REQUEST['order_nr'])) {
            $order = new WC_Order($_REQUEST['order_nr']);

            // Get country code
            if (method_exists($order, 'get_billing_country')) {
                $country = $order->get_billing_country();
            } elseif (isset($order->billing_country)) {
                $country = $order->billing_country;
            } else {
                $country = get_option('woocommerce_default_country');
            }

            // get billing email
            if (method_exists($order, 'get_billing_email')) {
                $billingEmail = $order->get_billing_email();
            } else {
                $billingEmail = $order->billing_email;
            }

            // get billing phone
            if (method_exists($order, 'get_billing_phone')) {
                $billingPhone = $order->get_billing_phone();
            } else {
                $billingPhone = $order->billing_phone;
            }

            // get order id
            $orderId = $this->get_order_number_from_order($order);

            // if such order exists
            $paramsArray = array(
                'website_id' => $this->website_id,
                'order_nr' => $orderId,
                'redirect_url' => $this->get_return_url($order),
                'web_service_url' => $this->get_api_url(),
                'standard' => 'opay_8.1',
                'country' => $country,
                'amount' => number_format($order->get_total(), 2, '', ''),
                'currency' => get_woocommerce_currency(),
                'c_email' => $billingEmail,
                'c_mobile_nr' => $billingPhone
            );

            if (!empty($_REQUEST['channel'])) {
                $paramsArray['pass_through_channel_name'] = $_REQUEST['channel'];
                $order->add_order_note(__('Customer chose payment method', 'opay-woocommerce') . ': ' . $_REQUEST['channel']);
            } else {
                $order->add_order_note(__('Customer redirected to external OPAY page', 'opay-woocommerce'));
            }

            if (!empty($_REQUEST['language'])) {
                $paramsArray['language'] = $_REQUEST['language'];
            } else {
                $paramsArray['language'] = $this->wp_language;
            }

            if ($this->test_mode == 'enabled') {
                $paramsArray['test'] = $this->user_id;
            }

            if (!isset($this->opay)) {
                $this->opay = new OpayGateway();
            }

            if ($this->signature_type == 'rsa_private_key') {
                $this->opay->setMerchantRsaPrivateKey($this->private_key);
                $this->opay->setOpayCertificate($this->opay_certificate);
            } else {
                $this->opay->setSignaturePassword($this->signature_password);
            }

            $this->appendMetadataToParams($paramsArray);
            $paramsArray = $this->opay->signArrayOfParameters($paramsArray);

            $url = sprintf('https://gateway.opay.lt/pay/?encoded=%s', $this->opay->convertArrayOfParametersToEncodedString($paramsArray));
            header(sprintf('Location: %s', $url));
        } else {
            $this->set_order_complete();
            echo 'OK';
        }
        exit();
    }

    public function add_checkout_css()
    {
        wp_enqueue_style('opay-style', plugins_url('css/style.css', __FILE__), array(), '', 'all');
    }

    public function add_admin_javascript()
    {
        // Get current tab/section
        $current_tab = empty($_GET['tab']) ? 'general' : sanitize_title($_GET['tab']);
        $current_section = empty($_REQUEST['section']) ? '' : sanitize_title($_REQUEST['section']);

        // Checking if we are in the checkout settings tab and in opay section
        //
        // Now Tab name is set to 'checkout', though in WC version 2.0.20
        // 'payment_gateways' is being used.
        // Now section name is set to class id but until version 2.6.0 it was
        // set to class name.
        if (
            in_array($current_tab, array('checkout', 'payment_gateways'))
            && in_array($current_section, array(strtolower(OPAY_CLASS_NAME), $this->id))
        ) {
            wp_enqueue_script('opay-woocommerce-admin', plugins_url() . DIRECTORY_SEPARATOR . 'opay-woocommerce' . DIRECTORY_SEPARATOR . 'js' . DIRECTORY_SEPARATOR . 'opay-woocommerce-admin.js', array('jquery'), 'v1.2');
        }
    }

    public function init_form_fields()
    {
        $this->form_fields = array(
            'enabled' => array(
                'title' => __('Enable / Disable', 'opay-woocommerce'),
                'label' => __('Enable OPAY Gateway', 'opay-woocommerce'),
                'type' => 'checkbox',
                'default' => 'no',
            ),
            'website_id' => array(
                'title' => '*' . __('Website ID', 'opay-woocommerce'),
                'type' => 'text',
                'custom_attributes' => array('required' => 'required')
            ),
            'test_mode' => array(
                'title' => __('Test mode', 'opay-woocommerce'),
                'type' => 'select',
                'options' => array(
                    'enabled' => __('Enabled', 'opay-woocommerce'),
                    'disabled' => __('Disabled', 'opay-woocommerce')
                ),
                'default' => 'disabled'
            ),
            'user_id' => array(
                'title' => '*' . __('User ID', 'opay-woocommerce'),
                'type' => 'text',
                'desc_tip' => __('Needed when Test mode is enabled', 'opay-woocommerce'),
                'custom_attributes' => array('required' => 'required')
            ),
            'signature_type' => array(
                'title' => __('Sign data with', 'opay-woocommerce'),
                'type' => 'select',
                'options' => array(
                    'rsa_private_key' => __('RSA private key', 'opay-woocommerce'),
                    'signature_password' => __('Signature password', 'opay-woocommerce')
                ),
                'default' => 'signature_password'
            ),
            'private_key' => array(
                'title' => '*' . __('Private key', 'opay-woocommerce'),
                'type' => 'textarea',
                'css' => 'height:200px;',
                'custom_attributes' => array('required' => 'required')
            ),
            'opay_certificate' => array(
                'title' => '*' . __('OPAY\'s certificate', 'opay-woocommerce'),
                'type' => 'textarea',
                'css' => 'height:200px;',
                'custom_attributes' => array('required' => 'required')
            ),
            'signature_password' => array(
                'title' => '*' . __('Signature password', 'opay-woocommerce'),
                'type' => 'text',
                'custom_attributes' => array('required' => 'required')
            ),
            'show_payment_methods' => array(
                'title' => __('Show payment methods', 'opay-woocommerce'),
                'type' => 'select',
                'options' => array(
                    'in_checkout_page_grouped' => __('In checkout page grouped', 'opay-woocommerce'),
                    'in_checkout_page' => __('In checkout page', 'opay-woocommerce'),
                    'in_external_opay_page' => __('In external OPAY page', 'opay-woocommerce')
                ),
                'default' => 'in_checkout_page_grouped'
            ),
            'height_of_banks_icons' => array(
                'title' => __('Height of banks\' icons', 'opay-woocommerce'),
                'type' => 'select',
                'options' => array(
                    '33' => '33px',
                    '49' => '49px',
                ),
                'default' => '33'
            ),
            //emisija.lt
            'enabled_wc_blocks' => array(
                'title' => __('Enable / Disable', 'opay-woocommerce'),
                'label' => __('Enable WC Blocks integration', 'opay-woocommerce'),
                'type' => 'checkbox',
                'default' => 'no',
            ),
            'enabled_opay_everywhere' => array(
                'title' => __('Enable / Disable', 'opay-woocommerce'),
                'label' => __('Enable OPAY on every page', 'opay-woocommerce') . '<span class="woocommerce-help-tip" data-tip="' . esc_attr('Enable or disable OPAY functionality on all pages. When disabled, it\'s only available on checkout and cart pages.') . '"></span>',  // Tooltip added here
                'type' => 'checkbox',
                'default' => 'yes',
            ),
            'use_remote_website_settings' => array(
                'title' => __('Enable / Disable', 'opay-woocommerce'),
                'label' => __('Enable OPAY default settings', 'opay-woocommerce') .
                    '<span class="woocommerce-help-tip" data-tip="' . esc_attr('When enabled, the title and description are retrieved from OPAY Gateway; if disabled, plugin-defined values are used.') . '"></span>',
                'type' => 'checkbox',
                'default' => 'yes',
            )
        );
    }

    public function process_payment($order_id)
    {
        global $wp;

        //emisija.lt
        if ($this->enabled_wc_blocks == 'yes') {
            if (empty($_POST['selected_channel'])) {
                throw new Exception(__('Please choose payment method', 'opay-woocommerce'));
            }
            if ($_POST['selected_channel'] == 'no-channel') {
                $_REQUEST['channel'] = '';
            } else {
                $_REQUEST['channel'] = $_POST['selected_channel'];
            }
        } else {
            if (empty($_REQUEST['channel']) && empty($_REQUEST['no-channel'])) {
                throw new Exception(__('Please choose payment method', 'opay-woocommerce'));
            }
        }

        $paramsArray['redirect-to-opay'] = '1';
        $paramsArray['order_nr'] = $order_id;

        if (!empty($_REQUEST['channel'])) {
            $paramsArray['channel'] = $_REQUEST['channel'];
        }

        $paramsArray['language'] = $this->wp_language;

        return array(
            'result' => 'success',
            'redirect' => add_query_arg($paramsArray, $this->get_api_url()),
        );
    }

    /**
     * Overidden function.
     * Older woocommerce versions use this instead of credit_card_form.
     */
    public function payment_fields()
    {
        $this->credit_card_form();
    }

    // overriding abstract-wc-payment-gateway.php method  credit_card_form()
    public function credit_card_form($args = array(), $fields = array())
    {
        $str = '';
        if (in_array($this->show_payment_methods, array('in_checkout_page_grouped', 'in_checkout_page'))) {
            $channelsArray = $this->get_channels();

            if (!empty($channelsArray)) {
                if ($this->show_payment_methods == 'in_checkout_page') {
                    $str .= '<ul style="display: block; list-style: outside none none; margin: 0; padding: 10px 0 0;">';
                }
                foreach ($channelsArray as $groupKey => $group) {
                    if ($this->show_payment_methods == 'in_checkout_page_grouped') {
                        $str .= '<label class="opay-payment-group-title opay-payment-group-title-' . $groupKey . '" style="padding-top:5px; font-weight:bold;">' . $group['group_title'] . '</label>';
                        $str .= '<ul class="opay-payment-group opay-payment-group-' . $groupKey . '" style="display: block; list-style: outside none none; margin: 0; padding: 10px 0 0;">';
                    }
                    foreach ($group['channels'] as $channelKey => $channel) {
                        if ($this->height_of_banks_icons == '33') {
                            $width = '110px';
                        } else if ($this->height_of_banks_icons == '49') {
                            $width = '170px';
                        }

                        if ($this->show_payment_methods == 'in_checkout_page') {
                            $groupClass = ' opay-payment-group-' . $groupKey;
                        } else {
                            $groupClass = '';
                        }

                        $str .= '
                            <li class="opay-payment-item' . $groupClass . ' opay-payment-item-' . $channelKey . '" style="box-sizing: border-box; display: block; float: left; list-style: outside none none; margin: 0; padding: 2px; position: relative; width:' . $width . ';">
                               <label for="in-' . $channelKey . '" style="border:solid 1px #d8d7d7; display: block; padding: 0 0 35px;" onMouseOver="this.style.backgroundColor=\'#e4e4e4\'"  onMouseOut="this.style.backgroundColor=\'\'">
                                  <span title="' . $channel['title'] . '" style="background-image:url(' . $channel['logo_urls']['color_' . $this->height_of_banks_icons . 'px'] . ');  background-position: center center; background-repeat: no-repeat; display: block; height: 65px; position: relative;">
                                     <input type="radio" id="in-' . $channelKey . '" value="' . $channelKey . '" name="channel" style="display: inline-block; left: 50%; margin-left: -5px; position: absolute; top: 70px;">
                                  </span>
                               </label>
                            </li>
                        ';
                    }
                    if ($this->show_payment_methods == 'in_checkout_page_grouped') {
                        $str .= '</ul>';
                        $str .= '<div style="clear: both; margin-bottom:10px;" class="opay-clear"></div>';
                    }
                }
                if ($this->show_payment_methods == 'in_checkout_page') {
                    $str .= '</ul>';
                    $str .= '<div style="clear: both; margin-bottom:10px;" class="opay-clear"></div>';
                }
            }
        }

        //emisija.lt
        if ($str != '') {
            if ($this->enabled_wc_blocks == 'yes') {
                echo "<div id='payment-opay-methods-list' style='display: none;'>" . $str . "</div>";
            } else {
                echo "<div id='payment-opay-methods-list-classic'>" . $str . "</div>";
            }
        } else {
            if ($this->enabled_wc_blocks == 'yes') {
                echo "<div id='payment-opay-methods-list' style='display: none;'><p>" . ($this->get_settings('payment_gateway_description') != '') ? '<p>' . $this->get_settings('payment_gateway_description') . '</p>' : '' . "<input type='hidden' name='no-channel' value='1'></div>";
            } else {
                echo "<div id='payment-opay-methods-list-classic'>";
                echo ($this->get_settings('payment_gateway_description') != '') ? '<p>' . $this->get_settings('payment_gateway_description') . '</p>' : '';
                echo '<input type="hidden" name="no-channel" value="1">';
                echo "</div>";
            }
        }
    }

    protected function get_settings($name = '')
    {
        $paramsArray = [
            'service_name' => 'getSettings',
            'website_id' => $this->get_option('website_id'),
        ];

        $session_var_name = 'website_settings' . md5(serialize($paramsArray));

        $languageCodeLower = strtolower($this->wp_language);
        $currentLangIsoCodeAlpha3 = strtolower($this->getLanguage([$languageCodeLower]));
        $countryCodeUpper = strtoupper($this->get_country_code());

        if (empty($this->get_session_var($session_var_name))) {
            if ($this->get_option('use_remote_website_settings') === 'no') {
                $paymentGatewayName = [
                    'lit' => 'El. bankininkystė ir kiti būdai',
                    'eng' => 'E-banking and other methods',
                    'rus' => 'Интернет-банки и другие способы',
                    'lav' => 'Internetbanka un citas metodes',
                    'est' => 'Internetipank ja muud meetodid'
                ];

                $paymentGatewayDescription = [
                    'lit' => 'Apmokėkite populiariausiais mokėjimo būdais',
                    'eng' => 'Pay by most popular payment methods',
                    'rus' => 'Оплатите самыми популярными способами оплаты ',
                    'lav' => 'Maksājiet, izmantojot vispopulārākas apmaksas metodes',
                    'est' => 'Makske, kasutades populaarseimaid maksemeetodeid'
                ];

                $languages = $this->get_all_languages();

                $settingsArray = [];
                foreach ($languages as $language) {
                    $langIsoCodeAlpha3 = strtolower($this->getLanguage([$language]));
                    if (in_array(strtolower($langIsoCodeAlpha3), $this->get_all_gw_languages($countryCodeUpper), true)) {
                        $settingsArray[$langIsoCodeAlpha3][$countryCodeUpper] = [
                            'payment_gateway_name' => $paymentGatewayName[$langIsoCodeAlpha3],
                            'payment_gateway_description' => $paymentGatewayDescription[$langIsoCodeAlpha3]
                        ];
                    }
                }

                $this->set_session_var($session_var_name, $settingsArray);
            } else {
                require_once __DIR__ . '/lib/opay_8.1.gateway.inc.php';

                if (!isset($this->opay)) {
                    $this->opay = new OpayGateway();
                }

                if ($this->get_option('signature_type') === 'rsa_private_key') {
                    $this->opay->setMerchantRsaPrivateKey($this->get_option('private_key'));
                    $this->opay->setOpayCertificate($this->get_option('opay_certificate'));
                } else {
                    $this->opay->setSignaturePassword($this->get_option('signature_password'));
                }

                try {
                    $settingsArray = [];
                    $paramsArray['language'] = $this->wp_language;
                    $paramsArray = $this->opay->signArrayOfParameters($paramsArray);
                    $array = $this->opay->webServiceRequest('https://gateway.opay.lt/api/websites/', $paramsArray);
                    if (!empty($array['response']['result'])) {
                        $settingsArray[$currentLangIsoCodeAlpha3][$countryCodeUpper] = $array['response']['result'];
                        $this->set_session_var($session_var_name, $settingsArray);
                    }
                } catch (OpayGatewayException $e) {
                    // do nothing
                }
            }
        }

        $settingsArray = $this->get_session_var($session_var_name);
        // Return specific or full settings
        if ($name !== '') {
            return isset($settingsArray[$currentLangIsoCodeAlpha3][$countryCodeUpper][$name]) ? $settingsArray[$currentLangIsoCodeAlpha3][$countryCodeUpper][$name] : '';
        }

        return $settingsArray[$currentLangIsoCodeAlpha3][$countryCodeUpper] ?: [];
    }

    protected function get_channels()
    {
        if (!isset($this->channels)) {
            require_once __DIR__ . '/lib/opay_8.1.gateway.inc.php';

            if (method_exists($this, 'get_order_total')) {
                $amount = number_format($this->get_order_total(), 2, '', '');
            } else {
                $amount = number_format($this->wc->cart->total, 2, '', '');
            }

            // Get country code
            if (isset($_REQUEST['country'])) {
                $country = $_REQUEST['country'];
            } else {
                $country = get_option('woocommerce_default_country');
            }

            $paramsArray = array(
                'website_id' => $this->website_id,
                'order_nr' => 'order nr',
                'redirect_url' => $this->get_return_url(),
                'web_service_url' => $this->get_api_url(),
                'standard' => 'opay_8.1',
                'country' => $country,
                'amount' => $amount,
                'currency' => get_woocommerce_currency()
            );

            $paramsArray['language'] = $this->wp_language;

            if ($this->test_mode == 'enabled') {
                $paramsArray['test'] = $this->user_id;
            }

            if (!isset($this->opay)) {
                $this->opay = new OpayGateway();
            }

            if ($this->signature_type == 'rsa_private_key') {
                $this->opay->setMerchantRsaPrivateKey($this->private_key);
                $this->opay->setOpayCertificate($this->opay_certificate);
            } else {
                $this->opay->setSignaturePassword($this->signature_password);
            }

            try {
                $paramsArray = $this->opay->signArrayOfParameters($paramsArray);
                $array = $this->opay->webServiceRequest('https://gateway.opay.lt/api/listchannels/', $paramsArray);

                if (!empty($array['response']['result'])) {
                    $this->channels = $array['response']['result'];
                }
            } catch (OpayGatewayException $e) {
                // do nothing
            }
        }

        return (isset($this->channels)) ? $this->channels : array();
    }

    public function validate_fields()
    {
        return true;
    }

    private function get_session_var($name)
    {
        if (isset($this->wc->session)) {
            if (method_exists($this->wc->session, 'get')) {
                return $this->wc->session->get($name, false);
            } else {
                if (isset($this->wc->session->$name)) {
                    return $this->wc->session->$name;
                }
            }
        }
        return false;
    }

    private function set_session_var($name, $value)
    {
        if (isset($this->wc->session)) {
            if (method_exists($this->wc->session, 'set')) {
                $this->wc->session->set($name, $value);
            } else {
                $this->wc->session->$name = $value;
            }
        }
    }

    private function get_order_number_from_order($order)
    {
        // check if a supported custom order numbering plugin is used
        if (class_exists('Alg_WC_Custom_Order_Numbers')) { // https://wordpress.org/plugins/custom-order-numbers-for-woocommerce
            if (preg_match('/^[a-zA-Z0-9ąčęėįšųūžĄČĘĖĮŠŲŪŽ,\.\s();\-]{1,40}$/u', $order->get_order_number())) {
                return $order->get_order_number();
            }
        }

        if (method_exists($order, 'get_id')) {
            return $order->get_id();
        } else {
            return $order->id;
        }
    }

    private function get_order_id_by_order_number($orderNumber)
    {
        // check if a supported custom order numbering plugin is used
        if (class_exists('Alg_WC_Custom_Order_Numbers')) { // https://wordpress.org/plugins/custom-order-numbers-for-woocommerce
            global $wpdb;
            $order = $wpdb->get_row(
                $wpdb->prepare(
                    "SELECT post_id FROM {$wpdb->postmeta} WHERE meta_value = %s AND meta_key = '_alg_wc_full_custom_order_number'",
                    $orderNumber
                )
            );
            if (isset($order->post_id)) {
                return $order->post_id;
            }
        }

        return $orderNumber;
    }

    private function get_api_url()
    {
        if (function_exists('pll_home_url')) {
            return add_query_arg('wc-api', __CLASS__, pll_home_url());
        }

        return $this->wc->api_request_url(__CLASS__);
    }

    private function appendMetadataToParams(&$paramsArray)
    {
        if (array_key_exists('metadata', $paramsArray)) {
            unset($paramsArray['metadata']);
        }

        //emisija.lt fix | New good metadata app_version define
        if (defined('WOOCOMMERCE_VERSION')) {
            $paramsArray['metadata']['app_version'] = WOOCOMMERCE_VERSION;
        }

        $paramsArray['metadata']['app_name'] = self::APP_NAME;
        $paramsArray['metadata']['widget_version'] = self::WIDGET_VERSION;
    }

    private function getLanguage($orderLanguage)
    {
        if (is_array($orderLanguage) === false) {
            return 'ENG';
        }

        if (in_array("lt", $orderLanguage)) {
            return 'LIT';
        }

        if (in_array("lv", $orderLanguage)) {
            return 'LAV';
        }

        if (in_array("ee", $orderLanguage)) {
            return 'EST';
        }

        if (in_array("ru", $orderLanguage)) {
            return 'RUS';
        }

        return 'ENG';
    }

    protected function get_country_code()
    {
        if (WC()->customer) {
            return WC()->customer->get_billing_country() ?: WC()->countries->get_base_country();
        }

        // Return a default country code if customer is not available
        return WC()->countries->get_base_country();
    }

    public function get_all_gw_languages($countryCode)
    {
        $countryCode = $this->validate_country($countryCode);

        $countriesSettingArr = self::CONFIG_COUNTRIES_SETTINGS;

        if (!empty($countriesSettingArr[$countryCode]['languages'])) {
            return $countriesSettingArr[$countryCode]['languages'];
        }

        return $countriesSettingArr['OTHER']['languages'];
    }

    public function validate_country($countryCode)
    {
        $supportedCountries = self::CONFIG_COUNTRIES_ALL;
        $countryCode = strtoupper($countryCode);
        if ($countryCode !== '' && in_array($countryCode, $supportedCountries)) {
            return $countryCode;
        }

        return $supportedCountries['default'];
    }

    function get_all_languages()
    {
        $language_list = []; // Unified result

        // Check if WPML is active and fetch its languages
        if (function_exists('icl_object_id')) {
            $languages = apply_filters('wpml_active_languages', null, ['skip_missing' => 0]);
            if ($languages) {
                foreach ($languages as $code => $language) {
                    $language_list[$code] = $language['native_name']; // Use native name for WPML
                }
            }
        }

        // Check if Polylang is active and fetch its languages
        if (function_exists('pll_get_the_languages')) {
            $languages = pll_get_the_languages(['fields' => 'name']);
            if ($languages) {
                foreach ($languages as $code => $name) {
                    $language_list[$code] = $name; // Use Polylang's language name
                }
            }
        }

        // Fallback to WordPress core languages
        $core_languages = get_available_languages(); // Returns an array of ISO 639-1 codes
        foreach ($core_languages as $language_code) {
            // If language is not already added, provide a default name
            if (!isset($language_list[$language_code])) {
                $language_list[$language_code] = ucfirst($language_code); // Placeholder name
            }
        }

        $language_list[substr(get_locale(), 0, 2)] = substr(get_locale(), 0, 2);

        return array_keys($language_list);
    }
}
