<?php

class CrossVersionsHelper
{
    const APP_NAME = 'PrestaShop';
    const WIDGET_VERSION = '1.6.5';

    public static function callback($module)
    {
        $opay = $module->getOpayGateway();
        $requestParams = $module->getEncodedValues();

        if (!empty($requestParams)) {
            if ($opay->verifySignature($requestParams)) {
                $db = \Db::getInstance();
                $prefix = _DB_PREFIX_;
                $dbName = _DB_NAME_;
                $request = "SELECT ENGINE FROM information_schema.TABLES WHERE TABLE_NAME = '{$prefix}orders' AND TABLE_SCHEMA = '{$dbName}'";
                $engine = $db->getRow($request)['ENGINE'];
                $orderid = $requestParams['order_nr'];
                $order = new Order($orderid);
                $orderIdEscaped = $db->escape($orderid, false);
                if ($engine == 'InnoDB') {
                    // Locking order until order state is changed to avoid possible double payment captures when user redirect and callback are happening at the exact same time
                    $db->execute('START TRANSACTION');
                    $request = "SELECT id_order, current_state FROM {$prefix}orders WHERE id_order = '{$orderIdEscaped}' LIMIT 0,1 FOR UPDATE";
                    $result = $db->executeS($request);
                    if (
                        isset($result[0]['current_state']) &&
                        !in_array($result[0]['current_state'], [
                            self::getPaymentState('OPAY_PENDING'),
                            self::getPaymentState('PS_OS_OUTOFSTOCK_UNPAID'),
                            self::getPaymentState('PS_OS_CANCELED'),
                            self::getPaymentState('PS_OS_ERROR'),
                            self::getPaymentState('PS_OS_PREPARATION')
                        ])
                    ) {
                        $db->execute('ROLLBACK');
                        exit('OK');
                    }
                }

                // Marking as payed only if current state is OPAY_PENDING || PS_OS_OUTOFSTOCK_UNPAID || PS_OS_CANCELED || PS_OS_ERROR || PS_OS_PREPARATION
                if (
                    (int) $requestParams['status'] == 1 &&
                    in_array((int) $order->getCurrentState(), [
                        self::getPaymentState('OPAY_PENDING'),
                        self::getPaymentState('PS_OS_OUTOFSTOCK_UNPAID'),
                        self::getPaymentState('PS_OS_CANCELED'),
                        self::getPaymentState('PS_OS_ERROR'),
                        self::getPaymentState('PS_OS_PREPARATION')
                    ])
                ) {
                    $order_amount = version_compare(_PS_VERSION_, '1.6', '<') ? $order->total_paid : $order->getOrdersTotalPaid();
                    $cart_currency = Currency::getCurrency($order->id_currency);

                    if (strtoupper($requestParams['p_currency']) != strtoupper($cart_currency['iso_code'])) {
                        if ($engine == 'InnoDB') {
                            $db->execute('ROLLBACK');
                        }
                        exit('Bad currency: ' . $requestParams['currency']);
                    }

                    if ((int) $requestParams['p_amount'] < (int) number_format(($order_amount * 100), 0, '', '')) {
                        if ($engine == 'InnoDB') {
                            $db->execute('ROLLBACK');
                        }
                        exit('Bad amount: ' . $requestParams['amount']);
                    }

                    $history = new OrderHistory();
                    $history->id_order = $orderid;
                    if (in_array((int) $order->getCurrentState(), [self::getPaymentState('OPAY_PENDING'), self::getPaymentState('PS_OS_CANCELED'), self::getPaymentState('PS_OS_ERROR'), self::getPaymentState('PS_OS_PREPARATION')])) {
                        $history->changeIdOrderState(self::getPaymentState('PS_OS_PAYMENT'), $orderid);
                    } else {
                        $history->changeIdOrderState(self::getPaymentState('PS_OS_OUTOFSTOCK_PAID'), $orderid);
                    }
                    $history->addWithemail(true, array(
                        'order_name' => $orderid,
                    ));

                    // Create object again to create an invoice
                    $order = new Order($orderid);
                    $order->payment = 'OPAY (' . $requestParams['p_channel'] . '_' . $requestParams['p_bank'] . ')';
                    $order->save();
                }
                // marking as canceled only if current status is OPAY_PENDING OR PS_OS_OUTOFSTOCK_UNPAID OR PS_OS_ERROR OR PS_OS_PREPARATION
                elseif (
                    (int) $requestParams['status'] == 0 &&
                    in_array((int) $order->getCurrentState(), [self::getPaymentState('OPAY_PENDING'), self::getPaymentState('PS_OS_OUTOFSTOCK_UNPAID'), self::getPaymentState('PS_OS_ERROR'), self::getPaymentState('PS_OS_PREPARATION')])
                ) {
                    $history = new OrderHistory();
                    $history->id_order = $orderid;
                    $history->changeIdOrderState(self::getPaymentState('PS_OS_CANCELED'), $orderid);
                    $history->addWithemail(true, array(
                        'order_name' => $orderid,
                    ));
                }
                if ($engine == 'InnoDB') {
                    $db->execute('COMMIT');
                }
                echo "OK";
            } else {
                echo 'invalid signature';
            }
        }

        exit();
    }

    public static function payment($module, $context, $smarty = null)
    {
        $currency = $context->currency;
        $language_code = Tools::strtoupper(Language::getIsoById((int) $context->language->id));
        $cart = $context->cart;
        $address = new Address((int) $cart->id_address_invoice);
        $country = new Country((int) $address->id_country);
        $customer = new Customer((int) $cart->id_customer);

        // Debug informacija apie konfigūraciją
        if ($module->debugmode) {
            $opayPendingStatus = Configuration::get('OPAY_PENDING');
            PrestaShopLogger::addLog(
                'Opay Payment Module: Starting payment process. OPAY_PENDING status: ' .
                    ($opayPendingStatus ? $opayPendingStatus : 'NOT SET') .
                    ', PS_VERSION: ' . _PS_VERSION_ .
                    ', Cart ID: ' . $cart->id,
                1,
                null,
                'Cart',
                $cart->id,
                true
            );
        }

        //em - fix to check prestashop version
        if (version_compare(_PS_VERSION_, '8.0.0', '>=')) {
            $total = $cart->getOrderTotal(true, Cart::BOTH);
        } else {
            $total = $cart->getOrderTotal(); // Default behavior in older versions
        }

        //em - add prestashop logger if total is zero or negative
        if ($total <= 0) {
            PrestaShopLogger::addLog(
                'Opay Payment Module: Invalid cart total amount: ' . $total . ' Cart ID: ' . $cart->id,
                3, // Severity level 3 (Warning)
                null,
                'Cart',
                $cart->id,
                true
            );
        }

        // Initialize order ID variable
        $orderId = null;

        // Patikriname order status prieš validateOrder
        $pendingStatus = Configuration::get('OPAY_PENDING');

        // Patikriname ar status egzistuoja ir ar jis valid
        $statusExists = false;

        if ($pendingStatus) {
            $orderState = new OrderState($pendingStatus);
            $statusExists = Validate::isLoadedObject($orderState) && !$orderState->deleted;
        }

        if (!$statusExists) {
            PrestaShopLogger::addLog(
                'Opay Payment Module: OPAY_PENDING status invalid or missing (ID: ' . $pendingStatus . '), attempting to recreate...',
                2,
                null,
                'Cart',
                $cart->id,
                true
            );

            /* Trying to add pending order state again if not exists */
            $order_pending = new OrderState();
            $order_pending->name = array_fill(0, 10, 'Awaiting OPAY payment');
            $order_pending->send_email = false;
            $order_pending->invoice = false;
            $order_pending->unremovable = false;
            $order_pending->logable = false;

            if (version_compare(_PS_VERSION_, '1.5.0.2', '>=')) {
                $order_pending->paid = false;
            }
            if (version_compare(_PS_VERSION_, '1.6.0.9', '>=')) {
                $order_pending->color = '#4169E1';
            } elseif (version_compare(_PS_VERSION_, '1.5', '>=')) {
                $order_pending->color = 'RoyalBlue';
            } else {
                $order_pending->color = 'lightblue';
            }
            if (version_compare(_PS_VERSION_, '1.5.0.15', '>=')) {
                $order_pending->module_name = 'opay';
            }

            if ($order_pending->add()) {
                copy(_PS_ROOT_DIR_ . '/modules/opay/logo.gif', _PS_ROOT_DIR_ . '/img/os/' . (int) $order_pending->id . '.gif');

                Configuration::updateValue('OPAY_PENDING', $order_pending->id);
                $pendingStatus = $order_pending->id;

                PrestaShopLogger::addLog(
                    'Opay Payment Module: Successfully recreated OPAY_PENDING status with ID: ' . $pendingStatus,
                    1,
                    null,
                    'Cart',
                    $cart->id,
                    true
                );
            } else {
                // Jei nepavyko sukurti, naudojame fallback
                $pendingStatus = Configuration::get('PS_OS_PREPARATION');
                if (!$pendingStatus) {
                    $pendingStatus = Db::getInstance()->getValue('
                    SELECT id_order_state
                    FROM ' . _DB_PREFIX_ . 'order_state
                    WHERE deleted = 0
                    AND delivery = 0
                    AND shipped = 0
                    AND paid = 0
                    LIMIT 1
                ');
                }

                PrestaShopLogger::addLog(
                    'Opay Payment Module: Failed to recreate OPAY_PENDING, using fallback status: ' . $pendingStatus,
                    3,
                    null,
                    'Cart',
                    $cart->id,
                    true
                );
            }
        }

        try {
            $module->validateOrder(
                $cart->id,
                $pendingStatus,
                $total,
                $module->displayName,
                null,
                array(),
                $currency->id,
                false,
                $cart->secure_key
            );

            // Tiesiogiai gauname sukurtą order ID
            $orderId = $module->currentOrder;

            // Papildoma patikra senoms versijoms (ypač 1.6.1.18)
            if (!$orderId) {
                // Bandome rasti paskutinį sukurtą užsakymą šiam krepšeliui
                $orderId = Db::getInstance()->getValue('
                SELECT id_order
                FROM ' . _DB_PREFIX_ . 'orders
                WHERE id_cart = ' . (int)$cart->id . '
                ORDER BY id_order DESC
            ');

                // Jei vis dar neturime order ID, bandome alternatyvų metodą
                if (!$orderId && method_exists('Order', 'getOrderByCartId')) {
                    $orderId = Order::getOrderByCartId($cart->id);
                }
            }

            // Debug informacija apie order ID
            if ($module->debugmode) {
                PrestaShopLogger::addLog(
                    'Opay Payment Module: After validateOrder - Order ID: ' .
                        ($orderId ? $orderId : 'STILL EMPTY') .
                        ', module->currentOrder: ' . ($module->currentOrder ? $module->currentOrder : 'EMPTY') .
                        ', Cart ID: ' . $cart->id,
                    1,
                    null,
                    'Cart',
                    $cart->id,
                    true
                );
            }
        } catch (Exception $e) {
            PrestaShopLogger::addLog(
                'Opay Payment Module: Order validation failed: ' . $e->getMessage() .
                    ', Cart ID: ' . $cart->id,
                3,
                null,
                'Cart',
                $cart->id,
                true
            );
            // TODO redirect client somewhere else
            return false;
        }

        // Patikriname ar turime validų order ID
        if (!$orderId) {
            PrestaShopLogger::addLog(
                'Opay Payment Module: Failed to get Order ID after validateOrder, Cart ID: ' . $cart->id,
                3,
                null,
                'Cart',
                $cart->id,
                true
            );
            // TODO redirect client somewhere else
            return false;
        }

        $opay = $module->getOpayGateway();

        $redirectUrl = self::getModuleLink(
            $module->name,
            'validation',
            array(),
            true
        );
        $webServiceUrl = self::getModuleLink(
            $module->name,
            'callback',
            array(),
            true
        );

        $paramsArray = array(
            'website_id' => $module->website_id,
            'order_nr' => $orderId, // Naudojame patikimai gautą order ID
            'redirect_url' => $redirectUrl,
            'web_service_url' => $webServiceUrl,
            'standard' => 'opay_8.1',
            'country' => $country->iso_code,
            'language' => $language_code,
            'amount' => (int) number_format($total, 2, '', ''),
            'currency' => $currency->iso_code,
            'c_email' => $customer->email
        );

        $address->phone_mobile = trim($address->phone_mobile);
        $address->phone = trim($address->phone);
        if ($address->phone_mobile != '') {
            $paramsArray['c_mobile_nr'] = $address->phone_mobile;
        } else if ($address->phone != '') {
            $paramsArray['c_mobile_nr'] = $address->phone;
        }

        if ($module->testmode) {
            $paramsArray['test'] = $module->opay_user_id;
        }

        //emisija.lt fix | New good metadata app_version define
        if (defined('_PS_VERSION_')) {
            $paramsArray['metadata']['app_version'] = _PS_VERSION_;
        }

        //em - add logger for ParamsArray request
        if ($module->debugmode) {
            PrestaShopLogger::addLog(
                'Opay Payment Module: payment() initiated. Order ID: ' . $orderId .
                    ', ParamsArray: ' . print_r($paramsArray, true),
                1,
                null,
                'Order', // Changed from 'Cart' to 'Order' since we now have order ID
                $orderId,
                true
            );
        }

        $channel = Tools::getValue('opay_channel');
        $channel = $channel ? $channel : Tools::getValue('type');

        if ($channel) {
            $paramsArray['pass_through_channel_name'] = $channel;
            $order = new Order($orderId); // Naudojame mūsų gautą order ID
            $order->payment = 'OPAY (' . $channel . ')';
            try {
                $order->save();
            } catch (Exception $e) {
                PrestaShopLogger::addLog(
                    'Opay Payment Module: Failed to update order payment method: ' . $e->getMessage() .
                        ', Order ID: ' . $orderId,
                    3,
                    null,
                    'Order',
                    $orderId,
                    true
                );
            }
        }

        self::appendMetadataToParams($paramsArray);
        $paramsArray = $opay->signArrayOfParameters($paramsArray);
        $url = sprintf('https://gateway.opay.lt/pay/?encoded=%s', $opay->convertArrayOfParametersToEncodedString($paramsArray));
        header(sprintf('Location: %s', $url));

        exit;
    }

    public static function validation($module, $context = null)
    {
        $opay = $module->getOpayGateway();
        $requestParams = $module->getEncodedValues($module);

        if (!empty($requestParams)) {
            if ($opay->verifySignature($requestParams)) {
                $ignore = false;
                $db = \Db::getInstance();
                $prefix = _DB_PREFIX_;
                $dbName = _DB_NAME_;
                $request = "SELECT ENGINE FROM information_schema.TABLES WHERE TABLE_NAME = '{$prefix}orders' AND TABLE_SCHEMA = '{$dbName}'";
                $engine = $db->getRow($request)['ENGINE'];
                $orderid = $requestParams['order_nr'];
                $order = new Order($orderid);
                $orderIdEscaped = $db->escape($orderid, false);

                if ($engine == 'InnoDB') {
                    // Locking order until order state is changed to avoid possible double payment captures when user redirect and callback are happening at the exact same time
                    $db->execute('START TRANSACTION');
                    $request = "SELECT id_order, current_state FROM {$prefix}orders WHERE id_order = '{$orderIdEscaped}' LIMIT 0,1 FOR UPDATE";
                    $result = $db->executeS($request);
                    if (
                        isset($result[0]['current_state']) &&
                        !in_array($result[0]['current_state'], [
                            self::getPaymentState('OPAY_PENDING'),
                            self::getPaymentState('PS_OS_OUTOFSTOCK_UNPAID'),
                            self::getPaymentState('PS_OS_CANCELED'),
                            self::getPaymentState('PS_OS_ERROR'),
                            self::getPaymentState('PS_OS_PREPARATION')
                        ])
                    ) {
                        $ignore = true;
                    }
                }

                // Marking as payed only if current state is OPAY_PENDING OR PS_OS_OUTOFSTOCK_PAID
                if (
                    !$ignore &&
                    (int) $requestParams['status'] == 1 &&
                    in_array((int) $order->getCurrentState(), [
                        self::getPaymentState('OPAY_PENDING'),
                        self::getPaymentState('PS_OS_OUTOFSTOCK_UNPAID'),
                        self::getPaymentState('PS_OS_CANCELED'),
                        self::getPaymentState('PS_OS_ERROR'),
                        self::getPaymentState('PS_OS_PREPARATION')
                    ])
                ) {
                    $order_amount = version_compare(_PS_VERSION_, '1.6', '<') ? $order->total_paid : $order->getOrdersTotalPaid();
                    $cart_currency = Currency::getCurrency($order->id_currency);


                    if ((int) $requestParams['p_amount'] >= (int) number_format(($order_amount * 100), 0, '', '') && strtoupper($requestParams['p_currency']) == strtoupper($cart_currency['iso_code'])) {
                        $history = new OrderHistory();
                        $history->id_order = $orderid;
                        if (in_array((int) $order->getCurrentState(), [self::getPaymentState('OPAY_PENDING'), self::getPaymentState('PS_OS_CANCELED'), self::getPaymentState('PS_OS_ERROR'), self::getPaymentState('PS_OS_PREPARATION')])) {
                            $history->changeIdOrderState(self::getPaymentState('PS_OS_PAYMENT'), $orderid);
                        } else {
                            $history->changeIdOrderState(self::getPaymentState('PS_OS_OUTOFSTOCK_PAID'), $orderid);
                        }
                        $history->addWithemail(true, array(
                            'order_name' => $orderid,
                        ));

                        // Create object again to create an invoice
                        $order = new Order($orderid);
                        $order->payment = 'OPAY (' . $requestParams['p_channel'] . '_' . $requestParams['p_bank'] . ')';
                        $order->save();
                    }
                }
                if ($engine == 'InnoDB') {
                    $db->execute('COMMIT');
                }
            }
        }

        // Redirect client to order confirmation page
        if (version_compare(_PS_VERSION_, '1.5.0.15', '>=')) {
            $customer = $order->getCustomer();
        } else {
            $customer = new Customer((int) $order->id_customer);
        }
        $queryString = http_build_query(array(
            'status' => $requestParams['status'],
            'id_cart' => $order->id_cart,
            'id_module' => $module->id,
            'id_order' => $orderid,
            'key' => $customer->secure_key,
        ));
        if (version_compare(_PS_VERSION_, '1.5', '>=')) {
            $confirmUri = 'index.php?controller=order-confirmation&';
        } else {
            $confirmUri = 'order-confirmation.php?';
        }
        Tools::redirect($confirmUri . $queryString);
    }

    public static function getShopDomainSsl($http = false, $entities = false)
    {
        if (method_exists('Tools', 'getShopDomainSsl')) {
            return Tools::getShopDomainSsl($http, $entities);
        } else {
            if (!($domain = Configuration::get('PS_SHOP_DOMAIN_SSL')))
                $domain = Tools::getHttpHost();
            if ($entities)
                $domain = htmlspecialchars($domain, ENT_COMPAT, 'UTF-8');
            if ($http)
                $domain = (Configuration::get('PS_SSL_ENABLED') ? 'https://' : 'http://') . $domain;
            return $domain;
        }
    }

    /**
     * Before v1.5.0.1 PrestaShop kept payment states defined as php constants
     * but later moved them to database. This function solves that.
     *
     * @param string $state State name
     * @return int|bool
     */
    public static function getPaymentState($state)
    {
        if (
            version_compare(_PS_VERSION_, '1.5.0.1', '>=')
            || $state == 'OPAY_PENDING'
        ) {
            return Configuration::get($state);
        } else {
            $constName = '_' . $state . '_';
            if (defined($constName)) {
                return constant($constName);
            }
        }
        return false;
    }

    /**
     * Creates a link to a module controller.
     *
     * @return string
     */
    public static function getModuleLink(
        $module,
        $controller = 'default',
        array $params = array(),
        $ssl = null,
        $idLang = null,
        $idShop = null,
        $relativeProtocol = false
    ) {
        if (version_compare(_PS_VERSION_, '1.5.0.5', '>=')) {
            $context = Context::getContext();
            $link = $context->link->getModuleLink(
                $module,
                $controller,
                $params,
                $ssl,
                $idLang,
                $idShop,
                $relativeProtocol
            );
        } elseif (version_compare(_PS_VERSION_, '1.5.0.1', '>=')) {
            $context = Context::getContext();
            $ssl = is_null($ssl) ? false : $ssl;
            $link = $context->link->getModuleLink(
                $module,
                $controller,
                $ssl,
                $idLang
            );
            if (!empty($params)) {
                $queryString = http_build_query($params);
                if (strpos($link, '?') === false) {
                    $link .= '?' . $queryString;
                } else {
                    $link .= '&' . $queryString;
                }
            }
        } else {
            $baseUri = self::getShopDomainSsl(true, true) . __PS_BASE_URI__;
            $link = $baseUri . 'modules/' . $module . '/' . $controller . '.php';

            if (!empty($params)) {
                $link .= '?' . http_build_query($params);
            }
        }

        return $link;
    }

    private static function appendMetadataToParams(&$paramsArray)
    {
        if (is_array($paramsArray) && array_key_exists('metadata', $paramsArray)) {
            unset($paramsArray['metadata']);
        }

        //emisija.lt fix | New good metadata app_version define
        if (defined('_PS_VERSION_')) {
            $paramsArray['metadata']['app_version'] = _PS_VERSION_;
        }

        $paramsArray['metadata']['app_name'] = self::APP_NAME;
        $paramsArray['metadata']['widget_version'] = self::WIDGET_VERSION;
    }
}
