<?php
if (is_file(DIR_SYSTEM.'library/opay/opay_8.1.gateway.inc.php')) {
    require_once DIR_SYSTEM.'library/opay/opay_8.1.gateway.inc.php';
} else {
    require_once DIR_SYSTEM.'library/vendor/opay/opay_8.1.gateway.inc.php';
}

class ControllerExtensionPaymentOpay extends Controller
{
    const APP_NAME = 'Opencart';
    const WIDGET_VERSION = '1.6.2';

    /**
     * @var int Order ID used when redirecting to opay
     */
    private $orderId;

    public function index()
    {
        $this->opay = new OpayGateway;
        $orderInfo = $this->getOrderInfo();

        if (version_compare(VERSION, '2.3', '>=')) {
            $this->load->language('extension/payment/opay');
        } else {
            $this->load->language('payment/opay');
        }

        if (version_compare(VERSION, '2.3', '>=')) {
            $routeAction = 'extension/payment/opay/confirm';
        } else {
            $routeAction = 'payment/opay/confirm';
        }
        $data['action']         = $this->url->link($routeAction, '', true);
        $data['button_confirm'] = $this->language->get('button_confirm');
        $data['order_id']       = $orderInfo['order_id'];

        $data['show_channels'] = $this->config->get('opay_show_channels');
        if (1 == $data['show_channels'] || 2 == $data['show_channels']) {
            $data['channels']                     = $this->getChannelsList();
            $data['payment_channel_icon_heigh']   = $this->config->get('opay_payment_channel_icon_height') ? $this->config->get('opay_payment_channel_icon_height') : '49';
            $data['please_select_payment_method'] = $this->language->get('text_please_select_payment_method');

            foreach ($data['channels'] as $channelGroupName => $channel) {
                $translationKey = 'opay_payment_group_description_'.$channelGroupName;
                // checking if such translation is defined in the language files
                // language->get() returns translation key if value is not found
                if (($translationValue = $this->language->get($translationKey)) != $translationKey) {
                    $data[$translationKey] = $translationValue;
                }
            }
        }

        $this->template = $this->getTemplate('opay.tpl');

        if (version_compare(VERSION, '2', '<')) {
            $this->data = array_merge($this->data, $data);
            $this->render();
        } else {
            return $this->load->view($this->template, $data);
        }
    }

    public function confirm() {
        $this->opay = new OpayGateway;

        // At this point we don't trust session data anymore. Order id is
        // submited along with channel name parameter in the checkout form.
        if (isset($_POST['order_id'])) {
            $this->orderId = $_POST['order_id'];
        }
        $order = $this->getOrderInfo();

        // Checking if this order info is valid. Order info is inserted into
        // database by previous request and here it must read from database. It
        // looks like some error happens in OC code when inserting order which
        // doesn't get caught and therefore we couldn't retrieve current order.
        // If payment method doesn't confirm order, it doesn't get listed in
        // admin panel.
        if (!isset($order['order_id'])) {
            $this->redirectMe($this->url->link('checkout/checkout'));
        }
        // discount coupon or gift card was used
        else if (!isset($order['total']) || $order['total'] <= 0)
        {
            $opayNewOrderId = $this->config->get('opay_new_order_id');
            // in case this functions gets called more than once
            if ((int)$order['order_status_id'] == 0) {
                $this->confirmOrder($order['order_id'], $opayNewOrderId);
            }
            $this->redirectMe($this->url->link('checkout/success', '', true));    
        }
        else
        {
            try 
            {

                // Set Opay signing keys
                $this->setKeys();

                // Get Opay parameters
                $paramsArray = $this->getParamsArray();
                $this->appendMetadataToParams($paramsArray);
                $paramsArray = $this->opay->signArrayOfParameters($paramsArray);

                // Get Opay url
                $url = sprintf(
                    'https://gateway.opay.lt/pay/?encoded=%s',
                    $this->opay->convertArrayOfParametersToEncodedString($paramsArray)
                );

                // in case this functions gets called more than once
                if ((int)$order['order_status_id'] == 0) {
                    // Add order status
                    $opayNewOrderId = $this->config->get('opay_new_order_id');
                    $this->confirmOrder($order['order_id'], $opayNewOrderId);

                    // Add chosen channel as a comment for order history
                    if (isset($_POST['channel'])) {
                        if (version_compare(VERSION, '2.3', '>=')) {
                            $this->load->language('extension/payment/opay');
                        } else {
                            $this->load->language('payment/opay');
                        }
                        $comment = $this->language->get('text_clicked_channel') . $_POST['channel'];
                        $this->updateOrder($order['order_id'], $opayNewOrderId, $comment, FALSE);
                    }
                }

                header(sprintf('Location: %s', $url));
            } catch (OpayGatewayException $e) {
                $this->log->write('Opay error: '. $e->getMessage() );
                $this->redirectMe($this->url->link('common/home'));
            }
        }
    }

    public function accept()
    {
        $statusReturned = $this->callback(true);

        if (isset($this->session->data['token'])) {
            $redirectParams = 'token='.$this->session->data['token'];
        } else {
            $redirectParams = '';
        }

        if ($statusReturned == 2) {
            if (version_compare(VERSION, '2.3', '>=')) {
                $route = 'extension/payment/opay/processing';
            } else {
                $route = 'payment/opay/processing';
            }
            $this->redirectMe($this->url->link($route, $redirectParams, true));
        } else {
            $this->redirectMe($this->url->link('checkout/success', $redirectParams, true));
        }
    }

    public function processing()
    {
        if (version_compare(VERSION, '2.3', '>=')) {
            $this->load->language('extension/payment/opay');
        } else {
            $this->load->language('payment/opay');
        }

        $this->document->setTitle($this->language->get('heading_title'));

        $data['heading_title'] = $this->language->get('heading_processing');
        $data['text_message']  = $this->language->get('text_payment_processing');

        $this->document->setTitle($this->language->get('heading_processing'));

        $data['breadcrumbs'] = array();

        $this->load->language('checkout/success');

        $data['breadcrumbs'][] = array(
            'href'      => $this->url->link('common/home'),
            'text'      => $this->language->get('text_home'),
            'separator' => false
        );

        $data['breadcrumbs'][] = array(
            'href'      => $this->url->link('checkout/cart'),
            'text'      => $this->language->get('text_basket'),
            'separator' => $this->language->get('text_separator')
        );

        $data['breadcrumbs'][] = array(
            'href'      => $this->url->link('checkout/checkout', '', 'SSL'),
            'text'      => $this->language->get('text_checkout'),
            'separator' => $this->language->get('text_separator')
        );

        $data['breadcrumbs'][] = array(
            'href'      => $this->url->link('payment/opay/processing'),
            'text'      => $this->language->get('text_processing'),
            'separator' => $this->language->get('text_separator')
        );

        $data['button_continue'] = $this->language->get('button_continue');
        $data['continue'] = $this->url->link('common/home');

        $this->template = $this->getTemplate('opay_processing.tpl');

        if (version_compare(VERSION, '2', '<')) {
            $this->data = array_merge($this->data, $data);
            $this->children = array(
                'common/column_left',
                'common/column_right',
                'common/content_top',
                'common/content_bottom',
                'common/footer',
                'common/header'
            );
            $this->response->setOutput($this->render(true));
        } else {
            $data['column_left'] = $this->load->controller('common/column_left');
            $data['column_right'] = $this->load->controller('common/column_right');
            $data['content_top'] = $this->load->controller('common/content_top');
            $data['content_bottom'] = $this->load->controller('common/content_bottom');
            $data['footer'] = $this->load->controller('common/footer');
            $data['header'] = $this->load->controller('common/header');
            $this->response->setOutput($this->load->view($this->template, $data));
        }
    }

    /**
     * @return int - Posible values {0, 1, 2}
     */
    public function callback($returnStatusOnly = false)
    {
        $this->opay = new OpayGateway;

        $this->load->model('checkout/order');
        $this->load->model('payment/opay');
        $engine = $this->model_payment_opay->getDbEngine();
        if (version_compare(VERSION, '2.3', '>=')) {
            $this->load->language('extension/payment/opay');
        } else {
            $this->load->language('payment/opay');
        }

        $opayPendingOrderId = $this->config->get('opay_new_order_id');
        $opayFinishedOrderId = $this->config->get('opay_finished_order_id');
        $opayCanceledOrderId = $this->config->get('opay_canceled_order_id');

        try {

            // Set Opay signing keys
            $this->setKeys();

            $response = array();

            if (isset($_POST['encoded']))
            {
                $response = $this->opay->convertEncodedStringToArrayOfParameters($_POST['encoded']);
            }
            else if (isset($_GET['encoded']))
            {
                $response = $this->opay->convertEncodedStringToArrayOfParameters($_GET['encoded']);
            }
            else if (isset($_POST['password_signature']) || isset($_POST['rsa_signature']))
            {
                $response = $_POST;
            }
            else if (isset($_GET['password_signature']) || isset($_GET['rsa_signature']))
            {
                $response = $_GET;
            }


            if (!empty($response))
            {
                if ($this->opay->verifySignature($response))
                {   
                    $message = '';
                    if($response['status'] == 1) {

                        $orderId = isset($response['order_nr']) ? $response['order_nr'] : null;
                        $order = $this->model_checkout_order->getOrder($orderId);
                        
                        if (empty($order)) {
                            throw new OpayGatewayException('Order with ID "'.$orderId.'" was not found.');
                        }
                        if ($engine == 'InnoDB') {
                            $this->db->query("START TRANSACTION");
                            // Locking order until order state is changed to avoid possible double payment captures when user redirect and callback are happening at the exact same time
                            $lockedStatus = $this->model_payment_opay->getOrderStatusIdAndLock($orderId);
                            if ($lockedStatus != $opayPendingOrderId && $lockedStatus != $opayCanceledOrderId) {
                                $this->db->query("ROLLBACK");
                                if ($returnStatusOnly) {
                                    return 1;
                                }
                                exit('OK');
                            }
                        }
                        // Get expected total amount in default currency
                        $orderTotal = intval(number_format($order['total'] * $order['currency_value'], 2, '', ''));
                        
                        
                        // here is the case when payment currency do not match orde currency 
                        // (possibly there was a currency conversion before transmitting payment data to OPAY)
                        if (strtoupper($order['currency_code']) != strtoupper($response['p_currency'])) 
                        {

                            $convertToCurrency = trim($this->config->get('opay_convert_order_currency_code')); // do't convert if empty value
        
                            // if transaction currency and actual payment currency match up (OPAY says it was payed the same currency you requested)
                            // if there is a setting telling us to convert order's currency before sending data to OPAY
                            // (the same setting tells us to convert the currency back to the original after we get a notification from OPAY) 
                            if (strtoupper($response['currency']) == strtoupper($response['p_currency']) && !empty($convertToCurrency) && $this->currency->has($convertToCurrency))
                            {
                                // here we compare order amount and actual payment amount taken from OPAY notification.
                                // we don't compare p_amount to amount taken from website order because 
                                if ($response['p_amount'] < $response['amount']) {
                                    throw new OpayGatewayException('Amount mismatch.: ' . $response['p_amount'] . ', expected: ' . $response['amount'] . '. Order ID: '.$orderId);
                                }
                                
                            }                    
                            else
                            {                                
                                throw new OpayGatewayException('Wrong currency. Expected '.$order['currency_code'].', but got '.$response['p_currency'].'. Order ID: '.$orderId);
                            }
                        }
                        else
                        {
                            if ($response['p_amount'] < $orderTotal) {
                                throw new OpayGatewayException('Amount mismatch: ' . $response['p_amount'] . ', expected: ' . $orderTotal . '. Order ID: '.$orderId);
                            }    
                        }
                        

                        // Change status if the payment was not registered
                        if ($order['order_status_id'] == $opayPendingOrderId OR $order['order_status_id'] == $opayCanceledOrderId) {
                            $message = $this->language->get('text_paid_by') . ' '. ucfirst($response['p_bank']);
                            $this->updateOrder($response['order_nr'], $opayFinishedOrderId, $message, FALSE);
                        }

                        if ($engine == 'InnoDB') {
                            $this->db->query('COMMIT');
                        }
                        
                        if ($returnStatusOnly) {
                            return 1;
                        }
                        exit('OK');
                    }
                    elseif($response['status'] == 2)
                    {
                        if ($returnStatusOnly) {
                            return 2;
                        }
                        exit('OK');
                    }
                    elseif($response['status'] == 0)
                    {
                        exit('OK');
                    }

                }
                else
                {
                    throw new OpayGatewayException('Wrong signature.');
                }
            }
            else
            {
                throw new OpayGatewayException('No direct access.');
            }
        } catch (OpayGatewayException $e) {
            $this->log->write('Opay error: '. $e->getMessage() );
            if ($returnStatusOnly)
            {
                return 0;
            }
            echo 'NOT OK' . PHP_EOL;
            exit('Error: ' . $e->getMessage());
        }
    }

    private function redirectMe($url) {
        if (version_compare(VERSION, '2', '<')) {
            $this->redirect($url);
        } else {
            $this->response->redirect($url);
        }
    }

    private function confirmOrder($order_id, $order_status_id, $comment = '', $notify = false) {

        if (version_compare(VERSION, '2', '<')) {
            $this->model_checkout_order->confirm($order_id, $order_status_id, $comment, $notify);
        } else {
            $this->model_checkout_order->addOrderHistory($order_id, $order_status_id, $comment, $notify);
        }
    }

    private function updateOrder($order_id, $order_status_id, $comment = '', $notify = false) {

        if (version_compare(VERSION, '2', '<')) {
            $this->model_checkout_order->update($order_id, $order_status_id, $comment, $notify);
        } else {
            $this->model_checkout_order->addOrderHistory($order_id, $order_status_id, $comment, $notify);
        }
    }

    private function setKeys() {

        $signatureType = $this->config->get('opay_signature_type');

        if ($signatureType == 'rsa')
        {
            $this->opay->setMerchantRsaPrivateKey($this->config->get('opay_rsa_signature'));
            $this->opay->setOpayCertificate($this->config->get('opay_certificate'));
        }
        else
        {
            $this->opay->setSignaturePassword($this->config->get('opay_password_sign'));
        }

    }

    private function getParamsArray() {

        $order = $this->getOrderInfo();

        $defaultCurrency = $this->config->get('config_currency');
        $convertToCurrency = trim($this->config->get('opay_convert_order_currency_code')); // do't convert if empty value
        
        // If customer can create order in currency which have currency value
        // different than 1, order total must be multiplied by currency value
        // to get the amount which customer sees. 
        $orderTotal = $order['currency_value'] != 1 ? $order['total'] * $order['currency_value'] : $order['total'];
        
        // if there is a setting telling us to convert order's currency before sending data to OPAY
        if (!empty($convertToCurrency) && $order['currency_code'] != $convertToCurrency && $this->currency->has($convertToCurrency))
        {
            // convert order total to EUR
            $orderTotal = $this->currency->convert($orderTotal, $order['currency_code'], $convertToCurrency);
            $currency = 'EUR';
        }
        else
        {
            $currency = $order['currency_code'];
        }
        $orderTotal = intval(number_format($orderTotal, 2, '', ''));

        if (version_compare(VERSION, '2.3', '>=')) {
            $redirectUrl   = $this->url->link('extension/payment/opay/accept', '', true);
            $webServiceUrl = $this->url->link('extension/payment/opay/callback', '', true);
        } else {
            $redirectUrl   = $this->url->link('payment/opay/accept', '', true);
            $webServiceUrl = $this->url->link('payment/opay/callback', '', true);
        }

        $paramsArray = array(
            'website_id'                => $this->config->get('opay_website_id'),
            'order_nr'                  => $order['order_id'],
            'redirect_url'              => $redirectUrl,
            'web_service_url'           => $webServiceUrl,
            'standard'                  => 'opay_8.1',
            'language'                  => $this->language->get('code'),
            'amount'                    => $orderTotal,
            'currency'                  => $currency,
            'country'                   => $order['payment_iso_code_2'],
            'test'                      => ($this->config->get('opay_test_mode')) ? $this->config->get('opay_user_id') : '',
            'c_email'                   => $order['email'],
            'c_mobile_nr'               => $order['telephone'],
            'pass_through_channel_name' => (isset($_REQUEST['channel'])) ? $_REQUEST['channel'] : '',
        );

        if (!empty($this->session->data['removed_channels'])) {
            $paramsArray['hide_channels'] = $this->session->data['removed_channels'];
            unset($this->session->data['removed_channels']);
        }

        return $paramsArray;
    }

    private function getChannelsList() {

        $paramsArray = $this->getParamsArray();

        // Set Opay signing keys
        $this->setKeys();

        $paramsArray = $this->opay->signArrayOfParameters($paramsArray);
        $channelsArray = $this->opay->webServiceRequest('https://gateway.opay.lt/api/listchannels/', $paramsArray);


        //////// >>>>>>>>>
        // if there are errors which lead channels to be removed, then we collect the names of the removed channels so we could pass them to hide_channels parameter when sending a request
        // It prevents such errors to occur again (No error emails will be sent to the seller)
        if (!empty($channelsArray['response']['errors']) && is_array($channelsArray['response']['errors'])) {

            $removedChannelsArray = array();
            foreach ($channelsArray['response']['errors'] as $errorArr){
                if (!empty($errorArr['data']['removed_channels']))
                {
                    $splitArray = preg_split('/(\s+|\t+|,)/', $errorArr['data']['removed_channels'], -1, PREG_SPLIT_NO_EMPTY);
                    $removedChannelsArray = array_merge($removedChannelsArray, $splitArray);
                }
            }

            if (!empty($removedChannelsArray))
            {
                $removedChannelsArray = array_unique($removedChannelsArray);
                $this->session->data['removed_channels'] = implode(',', $removedChannelsArray);
            }
        }
        // <<<<<<<<<

        if (!empty($channelsArray['response']['result'])) {
            return $channelsArray['response']['result'];
        }
        return array();
    }

    private function getTemplate($name)
    {
        $template = '';

        if (file_exists(DIR_TEMPLATE . $this->config->get('config_template') . '/template/payment/' . $name)) {
            $template = $this->config->get('config_template') . '/template/payment/' . $name;
        } else {
            if (version_compare(VERSION, '2.2', '>=')) {
                $template = 'payment/' . $name;
            } else {
                $template = 'default/template/payment/' . $name;
            }
        }
        return $template;
    }

    /**
     * Gets order info from checkout order model. If orderId is not set,
     * tries to get it from session.
     *
     * @return array
     */
    private function getOrderInfo()
    {
        if (!isset($this->orderInfo)) {
            if (!isset($this->orderId)) {
                $this->orderId = $this->session->data['order_id'];
            }
            $this->load->model('checkout/order');
            $this->orderInfo = $this->model_checkout_order->getOrder($this->orderId);
        }
        return $this->orderInfo;
    }

    private function appendMetadataToParams(&$paramsArray)
    {
        if (is_array($paramsArray) && array_key_exists('metadata', $paramsArray)) {
            unset($paramsArray['metadata']);
        }

        try {
            if (defined(VERSION)) {
                $paramsArray['metadata']['app_version'] = VERSION;
            }
        } catch (Exception $e) {
            // Do nothing to avoid any risks to break payment process
        }

        $paramsArray['metadata']['app_name'] = self::APP_NAME;
        $paramsArray['metadata']['widget_version'] = self::WIDGET_VERSION;
    }
}
